<?php


declare(strict_types=1);


namespace Atlas\SecurityManagerBundle\Twig\Components\User;

use Atlas\SecurityManagerBundle\Dto\User\UserRoleDto;
use Atlas\SecurityManagerBundle\Entity\User\User;
use Atlas\SecurityManagerBundle\Exception\Role\RoleException;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Atlas\SecurityManagerBundle\Form\User\UserRoleForm as Form;
use Atlas\SecurityManagerBundle\Repository\User\UserRoleRepository;
use Atlas\SecurityManagerBundle\Service\User\UserManager;
use Atlas\SecurityManagerBundle\Twig\Components\Concern\FlashHelperTrait;
use Atlas\SecurityManagerBundle\Twig\Components\Concern\SecurityUserHelperTrait;
use Doctrine\ORM\NonUniqueResultException;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\Form\Exception\RuntimeException;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Exception\UnprocessableEntityHttpException;
use Symfony\Component\OptionsResolver\Exception\InvalidOptionsException;
use Symfony\Component\Security\Core\Exception\AccessDeniedException;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveArg;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\ComponentToolsTrait;
use Symfony\UX\LiveComponent\ComponentWithFormTrait;
use Symfony\UX\LiveComponent\DefaultActionTrait;
use Symfony\UX\LiveComponent\Attribute\LiveProp;
use Symfony\UX\TwigComponent\Attribute\ExposeInTemplate;


/**
 * Inline listing + add/remove for user roles (with optional location scope).
 */
#[AsLiveComponent('user_role_form', template: '@SecurityManager/components/user/user_role_form.html.twig')]
final class UserRoleFormComponent
{
    use DefaultActionTrait;
    use ComponentToolsTrait;
    use ComponentWithFormTrait;
    use FlashHelperTrait;
    use SecurityUserHelperTrait;


    #[LiveProp(writable: false)]
    public User $user;

    public function __construct(
        private readonly UserRoleRepository $user_roles,
        private readonly UserManager $user_manager,
        private readonly FormFactoryInterface $form_factory,
        private readonly RequestStack $requests,
        private readonly Security $security
    ) {}

    /**
     * @return FormInterface
     * @throws InvalidOptionsException
     */
    protected function instantiateForm(): FormInterface
    {
        $dto = new UserRoleDto();

        return $this->form_factory->createNamed(
            'form',
            Form::class,
            $dto,
            [
                'csrf_protection' => true,
            ]
        );
    }

    #[ExposeInTemplate(name: 'roles')]
    public function getRoles(): array
    {
        return $this->user_roles->findForUserOrdered($this->user->id);
    }

    /**
     * @return void
     * @throws AccessDeniedException
     * @throws RuntimeException
     * @throws UnprocessableEntityHttpException
     */
    #[LiveAction]
    public function save(): void
    {

        if (!$this->security->isGranted('permission', [
            'permission' => 'usr.role',
            'user' => $this->user->id
        ])) {
            throw new AccessDeniedException();
        }

        $this->submitForm();

        /** @var UserRoleDto $dto */
        $dto = $this->getForm()->getData();

        $user = $this->getSecurityUser();

        try {
            $this->user_manager->addUserRole($this->user, $dto, $user->email);
            $this->resetForm();
            $this->flash('success', 'Role added to user', placement: 'local');
        } catch (RoleException|NotBlankException|NonUniqueResultException $e) {
            $this->flash('warning', $e->getMessage() ?: 'Could not add role', placement: 'local');
        }

    }

    /**
     * @param int $id
     * @param string $reason
     * @return void
     * @throws AccessDeniedException
     */
    #[LiveAction]
    public function remove(#[LiveArg] int $id, #[LiveArg] string $reason): void
    {
        if (!$this->security->isGranted('permission', [
            'permission' => 'usr.role',
            'user' => $this->user->id
        ])) {
            throw new AccessDeniedException();
        }

        $assignment = $this->user_roles->find($id);

        if (!$assignment || $assignment->user->id !== $this->user->id) {
            $this->flash('danger', 'User role cannot be found', placement: 'local');
            return;
        }

        $user = $this->getSecurityUser();

        try {
            $this->user_manager->removeUserRole($assignment, $reason, $user->email);
            $this->flash('success', 'Role removed from user', placement: 'local');
        } catch (NotBlankException) {
            //do nothing the modal should warn about this
        }

        $this->resetForm();
    }

}
