<?php

namespace Atlas\SecurityManagerBundle\Validator;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

/**
 * Validator for the UniqueEntityField constraint.
 *
 * Checks if a given field value is unique in the database for the specified entity.
 * Skips the uniqueness check when editing and the existing record is the same as the current one.
 */
class UniqueEntityFieldValidator extends ConstraintValidator
{
    public function __construct(private readonly EntityManagerInterface $entity_manager)
    {
    }

    /**
     * Validates the given value against the UniqueEntityField constraint.
     *
     * @param mixed      $value      The value to validate
     * @param Constraint $constraint The constraint instance
     */
    public function validate(mixed $value, Constraint $constraint): void
    {
        if (!$constraint instanceof UniqueEntityField) {
            throw new UnexpectedTypeException($constraint, UniqueEntityField::class);
        }

        if ($value === null || $value === '') {
            return;
        }

        $repo = $this->entity_manager->getRepository($constraint->entity);
        $existing = $repo->findOneBy([$constraint->field => $value]);

        if ($existing === null) {
            return;
        }

        $object  = $this->context->getObject();
        $idField = $constraint->id ?: 'id'; // default to "id"
        $studly  = str_replace(' ', '', ucwords(str_replace('_', ' ', $idField)));
        $getter  = 'get' . $studly;

        $existingId = null;
        if (method_exists($existing, $getter)) {
            $existingId = $existing->{$getter}();
        } elseif (method_exists($existing, 'getId')) {
            $existingId = $existing->getId();
        } elseif (property_exists($existing, $idField)) {
            $existingId = $existing->{$idField};
        }

        $objectId = null;
        if ($object !== null) {
            if (method_exists($object, $getter)) {
                $objectId = $object->{$getter}();
            } elseif (method_exists($object, 'getId')) {
                $objectId = $object->getId();
            } elseif (property_exists($object, $idField)) {
                $objectId = $object->{$idField};
            }
        }

        if ($existingId !== null && $objectId !== null && (string) $existingId === (string) $objectId) {
            return; // same record → skip violation
        }

        $this->context->buildViolation($constraint->message)
            ->setParameter('{{ value }}', (string) $value)
            ->addViolation();

    }
}
