<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Security\Provider;

use Atlas\SecurityManagerBundle\Entity\User\User as EntityUser;
use Atlas\SecurityManagerBundle\Repository\Audit\LoginRepository;
use Atlas\SecurityManagerBundle\Repository\Audit\PasswordRepository;
use Atlas\SecurityManagerBundle\Repository\User\UserRepository;
use Atlas\SecurityManagerBundle\Repository\User\UserRoleRepository;
use Atlas\SecurityManagerBundle\Security\User as SecurityUser;
use DateMalformedStringException;
use LogicException;
use SensitiveParameter;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Ldap\Exception\ConnectionException;
use Symfony\Component\Ldap\Exception\InvalidSearchCredentialsException;
use Symfony\Component\Ldap\Exception\LdapException;
use Symfony\Component\Ldap\LdapInterface;
use Symfony\Component\Security\Core\Exception\UnsupportedUserException;
use Symfony\Component\Security\Core\Exception\UserNotFoundException;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Core\User\UserProviderInterface;

/**
 * DB-first user provider with optional LDAP enrichment (DN, names, email).
 *
 * - Loads the account from the local DB (source of truth for permissions, etc.)
 * - If LDAP is configured, resolves the user's DN and extra attributes.
 * - Returns App\Security\User (SecurityUser) used by your authenticator.
 *
 * This does NOT extend Symfony's LdapUserProvider; it composes LdapInterface
 * and performs a minimal, robust LDAP lookup compatible with different servers.
 */
final readonly class UserProvider implements UserProviderInterface
{
    public function __construct(
        private UserRepository $users,
        private LoginRepository $logins,
        private PasswordRepository $passwords,
        private UserRoleRepository $user_roles,
        private LdapInterface $ldap,
        private RequestStack $requestStack,
        #[Autowire('%security.ldap.directory.base_dn%')] private string $base_dn,
        #[Autowire('%security.ldap.directory.search_dn%')] private ?string $search_dn = null,
        #[SensitiveParameter] #[Autowire('%security.ldap.directory.search_password%')] private ?string $search_password = null,
        #[Autowire('%security.ldap.directory.filter%')] private ?string $base_filter = '(objectClass=person)',
        #[Autowire(param: 'security.ldap.attributes')] private array $ldap_attributes = [],
    ) {
        // Validate required LDAP attribute mapping (generic to any LDAP schema)
        foreach (['username', 'givenname', 'surname', 'email'] as $k) {
            if (!array_key_exists($k, $this->ldap_attributes) || trim((string) $this->ldap_attributes[$k]) === '') {
                throw new LogicException(sprintf('security.ldap.attributes.%s must be configured.', $k));
            }
        }
    }

    /**
     * Load a user by username/email (DB-first), enrich from LDAP if possible.
     *
     * @param string $identifier
     * @return UserInterface
     * @throws UserNotFoundException
     */
    public function loadUserByIdentifier(string $identifier): UserInterface
    {
        $user = $this->users->findOneByUsernameOrEmail($identifier);
        if (!$user instanceof EntityUser) {
            throw new UserNotFoundException('Cannot find user');
        }

        $firstname = $user->firstname;
        $lastname  = $user->lastname;
        $email = $user->email;
        $dn = null;

        // Try to resolve DN + attributes from LDAP (tolerant of all failure modes)
        if($user->is_internal) {
            try {
                [$dn_found, $given, $sn, $mail] = $this->resolveLdapAttributes($user->username);

                if ($dn_found !== null) {
                    $dn = $dn_found;
                    $firstname = $given ?? $firstname;
                    $lastname = $sn ?? $lastname;
                    $email = $mail ?? $email;
                }

            } catch (LdapException|ConnectionException|InvalidSearchCredentialsException) {
                // Ignore: fall back to DB-only (authenticator will skip LdapBadge)
            }
        }

        $failed_count = $this->logins->findLastLogin($user->id)?->failed_count ?? 0;
        $password = $this->passwords->getLatestPassword($user->id)?->password;
        $permissions = $this->user_roles->findUserPermissions($user->id);

        return new SecurityUser(
            $user->id,
            $firstname,
            $lastname,
            $user->username,
            $email,
            password: $password,
            dn: $dn,
            internal: $user->internal,
            locked: $user->is_locked,
            validated: $user->is_validated,
            failedCount: $failed_count,
            permissions: $permissions
        );
    }

    /**
     * @param mixed $username
     * @return UserInterface
     * @throws DateMalformedStringException
     * @deprecated Since Symfony 5.3. Use loadUserByIdentifier().
     */
    public function loadUserByUsername(string $username): UserInterface
    {
        return $this->loadUserByIdentifier((string) $username);
    }

    public function refreshUser(UserInterface $user): UserInterface
    {
        if (!$user instanceof SecurityUser) {
            throw new UnsupportedUserException(sprintf('Invalid user class "%s".', $user::class));
        }

        $request = $this->requestStack->getCurrentRequest();
        $session = $request?->getSession();

        if (!$session) {
            return $user;
        }

        // consume the flag once
        if ((int) $session->get('auth.force_perm_refresh', 0) === 1) {
            $session->remove('auth.force_perm_refresh');

            return $user->withUpdatedPermissions(
                $this->user_roles->findUserPermissions($user->id)
            );
        }

        return $user;
    }

    /**
     * @param string $class
     * @return bool
     */
    public function supportsClass(string $class): bool
    {
        return SecurityUser::class === $class;
    }

    /**
     * Resolve LDAP DN and common attributes for a username.
     *
     * Robust across different LDAP servers:
     * - Binds with service account if provided, else anonymous bind.
     * - Escapes filter safely and wraps base filter if needed.
     * - Returns [null,null,null,null] when 0 or multiple entries.
     *
     * @param string $username
     * @return array{0:?string,1:?string,2:?string,3:?string} [dn, givenname, surname, email]
     * @throws LdapException
     * @throws ConnectionException
     * @throws InvalidSearchCredentialsException
     */
    private function resolveLdapAttributes(string $username): array
    {
        // Bind using provided service account if configured, else anonymous.
        if ($this->search_dn !== null && $this->search_dn !== '') {
            $this->ldap->bind($this->search_dn, (string) $this->search_password);
        } else {
            $this->ldap->bind();
        }

        $uid_attr = trim((string) $this->ldap_attributes['username']);
        $gn_attr = trim((string) $this->ldap_attributes['givenname']);
        $sn_attr = trim((string) $this->ldap_attributes['surname']);
        $mail_attr = trim((string) $this->ldap_attributes['email']);

        // Build effective base filter
        $base = trim($this->base_filter ?? '');
        if ($base === '') {
            $base = '(objectClass=person)';
        }
        if ($base[0] !== '(') {
            $base = '(' . $base . ')';
        }

        // (&<base>(uid=<escaped>))
        $escaped_username = $this->ldap->escape($username, flags: LdapInterface::ESCAPE_FILTER);
        $filter = sprintf('(&%s(%s=%s))', $base, $uid_attr, $escaped_username);

        $cursor = $this->ldap->query($this->base_dn, $filter, [
            'maxItems' => 2,   // client-side iteration cap
            'sizeLimit' => 2,   // server-side cap (if supported)
            'timeout' => 5,   // seconds (optional)
        ])->execute();

        if ($cursor->count() !== 1) {
            return [null, null, null, null];
        }

        $entry = $cursor->toArray()[0];

        $dn = $entry->getDn();
        $gn = $entry->getAttribute($gn_attr)[0] ?? null;
        $sn = $entry->getAttribute($sn_attr)[0] ?? null;
        $mail = $entry->getAttribute($mail_attr)[0] ?? null;

        return [$dn, $gn, $sn, $mail];
    }
}
