<?php

declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Security;

use InvalidArgumentException;
use SensitiveParameter;
use Symfony\Component\Security\Core\User\PasswordAuthenticatedUserInterface;
use Symfony\Component\Security\Core\User\UserInterface;

class User implements UserInterface, PasswordAuthenticatedUserInterface
{
    //read-only only set by methods or constructor
    private(set) int $id;
    private(set) string $identifier = '';
    private(set) ?string $dn = null;
    private(set) ?string $password = null; //can't make this a hook, the get method later is required by Symfony
    private(set) string $username;
    private(set) string $email;
    private(set) string $firstname = '';
    private(set) string $lastname = '';
    private(set) bool $internal = false;
    private(set) bool $locked = true;
    private(set) bool $validated = false;
    private(set) int $failed_count = 0;

    // Backing store has the default
    private array $roles_raw = [];

    // Public API stays the same name: $user->roles
    public array $roles {
        get {
            $roles = $this->roles_raw;
            $roles[] = 'ROLE_USER';
            return array_values(array_unique($roles));
        }
        set {
            $this->roles_raw = $value ?? [];
        }
    }
    public array $permissions = [];

    //can not get or set only for use in methods

    //virtual variables
    public string $full_name {
        get => $this->firstname . ' ' . $this->lastname;
    }
    public string $inverted_name {
        get => $this->lastname . ', ' . $this->firstname;
    }
    public string $initials {
        get => $this->firstname[0] . $this->lastname[0];
    }
    public string $initial_lastname {
        get => $this->firstname[0] . ' ' . $this->lastname;
    }
    public bool $is_internal {
        get => $this->internal;
    }
    public bool $is_validated {
        get => $this->validated;
    }
    public bool $is_locked {
        get => $this->locked;
    }

    /**
     * @param int $id
     * @param string $firstname
     * @param string $lastname
     * @param string $username
     * @param string $email
     * @param string|null $password
     * @param string|null $dn
     * @param bool $internal
     * @param bool $locked
     * @param bool $validated
     * @param int $failedCount
     * @param array $roles
     * @param array $permissions
     */
    public function __construct(
        int $id,
        string $firstname,
        string $lastname,
        string $username,
        string $email,
        #[SensitiveParameter] ?string $password = null,
        ?string $dn = null,
        bool $internal = false,
        bool $locked = false,
        bool $validated = false,
        int $failedCount = 0,
        array $roles = [],
        array $permissions = [],
    ) {

        if (empty($firstname = mb_trim($firstname, encoding: 'UTF-8'))) {
            throw new InvalidArgumentException('Firstname should not be empty');
        }

        if (empty($lastname = mb_trim($lastname, encoding: 'UTF-8'))) {
            throw new InvalidArgumentException('Lastname should not be empty');
        }

        if (empty($username = mb_trim($username, encoding: 'UTF-8'))) {
            throw new InvalidArgumentException('Username should not be empty');
        }

        if (empty($email = mb_trim($email, encoding: 'UTF-8'))) {
            throw new InvalidArgumentException('Email should not be empty');
        }

        $this->id = $id;
        $this->identifier = $email;
        $this->internal = $internal;
        $this->dn = $dn;
        $this->password = $password;
        $this->firstname = $firstname;
        $this->lastname = $lastname;
        $this->username = $username;
        $this->email = $email;
        $this->locked = $locked;
        $this->validated = $validated;
        $this->failed_count = $failedCount;
        $this->roles = $roles;
        $this->permissions = $permissions;
    }

    //required methods for Symfony to function

    /**
     * A visual identifier that represents this user.
     *
     * @see UserInterface
     */
    public function getUserIdentifier(): string
    {
        return $this->identifier;
    }

    /**
     * @return string
     */
    public function getPassword(): string
    {
        return $this->password ?? '';
    }

    /**
     * @return array
     */
    public function getRoles(): array
    {
        return $this->roles;
    }

    /**
     * @see UserInterface
     */
    public function eraseCredentials(): void
    {
        //not used
    }

    /**
     * @param array $perms
     * @return $this
     */
    public function withUpdatedPermissions(array $perms): self
    {
        // clone and set new permissions in the clone
        $clone = clone $this;
        $clone->permissions = $perms;
        return $clone;
    }
}
