<?php
declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Service\Security;

use Atlas\SecurityManagerBundle\Exception\Security\InvalidClaimException;
use Atlas\SecurityManagerBundle\Exception\Security\KeyException;
use Atlas\SecurityManagerBundle\Exception\Validation\NotBlankException;
use Firebase\JWT\JWT;
use OpenSSLAsymmetricKey;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Uid\Uuid;

final readonly class JwtTokenGenerator
{
    /** @var array<string,bool> */
    private const array RESERVED = [
        'iss' => true, 'aud' => true, 'iat' => true, 'nbf' => true, 'exp' => true,
    ];

    private const int DEFAULT_TTL = 3600;
    private OpenSSLAsymmetricKey $private_key;

    /**
     * @param string $private_key_path
     * @param string $jwt_iss
     * @param string $jwt_aud
     * @param string $kid
     * @throws KeyException
     */
    public function __construct(
        #[Autowire('%security.base.specs_dir%/keys/private.pem')] string $private_key_path,
        #[Autowire('%security.jwt.iss%')]private string $jwt_iss,
        #[Autowire('%security.jwt.aud%')]private string $jwt_aud,
        #[Autowire('%security.jwt.kid%')]private string $kid = 'public'
    )
    {
        $private_pem = file_get_contents($private_key_path);
        if ($private_pem === false) {
            throw new KeyException('Private key could not be read');
        }

        $key = openssl_pkey_get_private($private_pem);

        if ($key === false) {
            throw new KeyException('Private key invalid');
        }

        $this->private_key = $key;
    }

    /**
     * @param array<string,mixed> $claims
     * @param int|null $ttlSeconds
     * @param int|null $expOverride
     * @return string
     * @throws InvalidClaimException
     */
    public function issue(array $claims, ?int $ttlSeconds = null, ?int $expOverride = null): string
    {
        $this->assertNoReservedCollision($claims);

        $now = time();
        $exp = $expOverride ?? ($now + ($ttlSeconds ?? self::DEFAULT_TTL));

        $payload = [
                'iss' => $this->jwt_iss,
                'aud' => $this->jwt_aud,
                'iat' => $now,
                'nbf' => $now,
                'exp' => $exp,
            ];

        $payload = array_merge($payload, $claims);

        return JWT::encode($payload, $this->private_key, 'ES256', $this->kid);
    }


    /**
     * @param string $username
     * @param int $ttlSeconds
     * @return string
     * @throws InvalidClaimException
     * @throws NotBlankException
     */
    public function issuePasswordReset(string $username, int $ttlSeconds = 604800): string
    {
        $username = mb_trim($username, encoding: 'UTF-8');
        if(mb_strlen($username, encoding: 'UTF-8') === 0) {
            throw new NotBlankException('Username cannot be empty');
        }

        $claims = [
                'purpose' => 'pwd_set',
                'sub' => $username
            ];

        return $this->issue($claims, ttlSeconds: $ttlSeconds);
    }

    /**
     * Issue a signed JWT granting access to a single survey link.
     *
     * Token validity is controlled by TTL; revocation is enforced via DB lookup on jti.
     *
     * @param Uuid $link Survey link UUID (jti)
     * @param int $ttlSeconds Defaults to 7 days (604800)
     */
    public function issueSurveyAccess(Uuid $link, int $ttlSeconds = 604800): string
    {
        $claims = [
            'purpose' => 'survey',
            'jti' => $link->toRfc4122(),
        ];

        return $this->issue($claims, ttlSeconds: $ttlSeconds);
    }

    /**
     * @param array<string,mixed> $claims
     * @throws InvalidClaimException
     */
    private function assertNoReservedCollision(array $claims): void
    {
        foreach ($claims as $k => $_) {
            if (isset(self::RESERVED[$k])) {
                throw new InvalidClaimException(sprintf('Claim "%s" is reserved and cannot be set manually.', $k));
            }
        }
    }
}
