<?php declare(strict_types=1);

namespace Atlas\SecurityManagerBundle\Service\Security;

/**
 * Enforces password policy and returns plain-English messages.
 *
 * Rules:
 * - Length: 8–20 characters
 * - Must include uppercase, lowercase, number
 * - Must include one of: ! @ # $ % ^ & *
 * - No dots (.) and no newlines
 * - Spaces allowed
 * - Control characters are not allowed
 */
final class PasswordPolicy
{
    /** @var string */
    private const string REGEX = '/(?=.*\d)(?=.*[!@#$%^&*]+)(?![.\n])(?=.*[A-Z])(?=.*[a-z]).*/';

    /**
     * @param string $password
     * @return array<int,string> Empty if valid; otherwise list of unmet rules.
     */
    public function validate(string $password): array
    {
        $errors = [];

        $length = mb_strlen($password);
        if ($length < 8) {
            $errors[] = 'Password is too short. It must be at least 8 characters.';
        }
        if ($length > 20) {
            $errors[] = 'Password is too long. It must be no more than 20 characters.';
        }

        if ($this->containsControlChars($password)) {
            $errors[] = 'Control characters are not allowed.';
        }

        if (!preg_match(self::REGEX, $password)) {
            if (!preg_match('/[A-Z]/u', $password)) {
                $errors[] = 'Add an uppercase letter. Include at least one A–Z.';
            }
            if (!preg_match('/[a-z]/u', $password)) {
                $errors[] = 'Add a lowercase letter. Include at least one a–z.';
            }
            if (!preg_match('/\d/u', $password)) {
                $errors[] = 'Add a number. Include at least one digit (0–9).';
            }
            if (!preg_match('/[!@#$%^&*]/u', $password)) {
                $errors[] = 'Add a special character. Include at least one of: ! @ # $ % ^ & *.';
            }
            if (mb_strpos($password, '.') !== false) {
                $errors[] = 'Dots are not allowed. Remove any "." characters.';
            }
            if (preg_match("/[\r\n]/u", $password)) {
                $errors[] = 'Newlines are not allowed. Enter the password on a single line.';
            }
        }

        return $errors;
    }

    private function containsControlChars(string $password): bool
    {
        // Deny ASCII control chars U+0000–U+001F and U+007F; spaces are allowed.
        return (bool) preg_match('/[\x00-\x1F\x7F]/', $password);
    }
}
