<?php

namespace Atlas\SecurityManagerBundle\Validator;

use Attribute;
use Symfony\Component\Validator\Constraint;

/**
 * Constraint to ensure a field value is unique within a specific entity.
 *
 * Example:
 * #[UniqueEntityField(entity: User::class, field: 'email')]
 *
 * @Annotation
 * @Target({"PROPERTY", "METHOD", "ANNOTATION"})
 */
#[Attribute]
final class UniqueEntityField extends Constraint
{
    private(set) string $entity;
    private(set) string $field;
    private(set) ?string $id;
    private(set) string $message;

    /**
     * UniqueEntityField constructor.
     *
     * @param string $entity Fully-qualified class name of the entity.
     * @param string $field The property/column to check for uniqueness.
     * @param string|null $id Optional identifier field to exclude from check (for editing).
     * @param string $message The error message when a duplicate is found.
     * @param array|null $groups Validation groups.
     * @param mixed|null $payload Custom payload passed to the constraint.
     */
    public function __construct(
        string $entity,
        string $field,
        ?string $id = null,
        string $message = 'The value "{{ value }}" is already used.',
        ?array $groups = null,
        mixed $payload = null
    ) {
        parent::__construct([], $groups, $payload);

        $this->entity = $entity;
        $this->field = $field;
        $this->id = $id;
        $this->message = $message;
    }

    /**
     * Returns the validator service class name for this constraint.
     *
     * @return string
     */
    public function validatedBy(): string
    {
        return UniqueEntityField::class . 'Validator';
    }
}
